﻿function Show-Menu {
    Clear-Host
    Write-Host "=== 共享桌面和RD网关配置工具 ==="
    Write-Host "1. 配置网络"
    Write-Host "2. 加入域并设置主机名"
    Write-Host "3. 安装 RDSH 角色和授权（适用于Windows Server）"
    Write-Host "4. 安装RDPWrapper启用多用户（适用于Windows 10/11）"
    Write-Host "5. 添加 VDI Users 群组"
    Write-Host "6. 配置RD Gateway"
    Write-Host "Q. 退出"
    Write-Host ""
}

function Set-Network {
    Write-Host "配置网络..." -ForegroundColor Yellow
    $IPAddress = Read-Host "输入 IP 地址"
    $Gateway = Read-Host "输入网关"
    $DNS = Read-Host "输入 DNS 服务器"
    
    $Adapter = Get-NetAdapter | Where-Object Status -eq 'Up' | Select-Object -First 1
    Remove-NetIPAddress -InterfaceAlias $Adapter.Name -Confirm:$false -ErrorAction SilentlyContinue
    Remove-NetRoute -InterfaceAlias $Adapter.Name -Confirm:$false -ErrorAction SilentlyContinue
    New-NetIPAddress -InterfaceAlias $Adapter.Name -IPAddress $IPAddress -PrefixLength 24 -DefaultGateway $Gateway
    Set-DnsClientServerAddress -InterfaceAlias $Adapter.Name -ServerAddresses $DNS
    
    Write-Host "网络配置完成" -ForegroundColor Green
}

function Join-ADDomain {
    Write-Host "加入域并设置主机名..." -ForegroundColor Yellow
    $DomainName = Read-Host "输入域的名称（比如dora.cloud)"
    $ComputerName = Read-Host "输入新的计算机名称"
    
    Add-Computer -DomainName $DomainName -NewName $ComputerName -Credential (Get-Credential) -Force -Restart
    Write-Host "已加入域并设置新主机名，系统将自动重启" -ForegroundColor Green
}

function Install-RDSRoles {
    Write-Host "正在安装 RDSH 角色和授权..." -ForegroundColor Yellow
    Install-WindowsFeature -Name RDS-RD-Server, RDS-Licensing -IncludeManagementTools
    Set-ItemProperty -Path 'HKLM:\SYSTEM\CurrentControlSet\Control\Terminal Server' -Name "fDenyTSConnections" -Value 0
    
    # 修正防火墙规则名称
    Enable-NetFirewallRule -DisplayGroup "远程桌面" -ErrorAction SilentlyContinue
    Enable-NetFirewallRule -DisplayGroup "Remote Desktop" -ErrorAction SilentlyContinue
    
    Write-Host "RDS 角色安装完成，需要重启生效" -ForegroundColor Green
    Write-Host "重启后请再次运行此脚本继续其他配置" -ForegroundColor Yellow
}

function Install-RDPWrapper {
    Write-Host "`n=== 安装 RDPWrapper ===" -ForegroundColor Cyan
    
    # 使用脚本所在目录查找文件
    $scriptDirectory = $PSScriptRoot
    $localZip = Join-Path $scriptDirectory "RDPWrapper.1.6.2.zip"
    
    Write-Host "正在在以下位置查找文件: $localZip" -ForegroundColor Yellow
    
    if (-not (Test-Path $localZip)) {
        Write-Host "错误: 在脚本目录下找不到 RDPWrapper.1.6.2.zip" -ForegroundColor Red
        Write-Host "请确保 RDPWrapper.1.6.2.zip 文件与脚本在同一目录中" -ForegroundColor Yellow
        return
    }

    Write-Host "找到 RDPWrapper.1.6.2.zip 文件，开始安装..." -ForegroundColor Green
    
    # 设置路径
    $extractPath = "$env:TEMP\RDPWrap"

    # 创建临时目录
    New-Item -Path $extractPath -ItemType Directory -Force | Out-Null

    # 解压文件
    Write-Host "正在解压文件..." -ForegroundColor Yellow
    Expand-Archive -Path $localZip -DestinationPath $extractPath -Force

    # 安装 RDPWrapper
    Write-Host "正在安装 RDPWrapper..." -ForegroundColor Yellow
    $originalLocation = Get-Location
    Set-Location "$extractPath"
    .\install.bat

    # 等待安装完成
    Start-Sleep -Seconds 5

    # 更新配置文件
    $iniUrl = "https://raw.githubusercontent.com/sebaxakerhtc/rdpwrap.ini/master/rdpwrap.ini"
    $iniPath = "C:\Program Files\RDP Wrapper\rdpwrap.ini"

    Write-Host "正在更新配置文件..." -ForegroundColor Yellow
    try {
        Invoke-WebRequest -Uri $iniUrl -OutFile $iniPath
        Write-Host "配置文件更新成功" -ForegroundColor Green
    }
    catch {
        Write-Host "警告: 无法下载最新配置文件，使用默认配置" -ForegroundColor Yellow
    }

    # 重启服务
    Write-Host "正在配置服务..." -ForegroundColor Yellow
    .\RDPWinst.exe -r

    # 返回原始目录
    Set-Location $originalLocation

    # 清理临时文件
    Write-Host "清理临时文件..." -ForegroundColor Yellow
    Remove-Item $extractPath -Recurse -Force

    Write-Host "`n安装完成！" -ForegroundColor Green
}

function Add-VDIUserGroup {
    net localgroup "Remote Desktop Users" "VDI Users" /add
    Write-Host "VDI Users 群组已添加到 Remote Desktop Users" -ForegroundColor Green
}


function Install-Configure-RDGateway {
    param(
        [string]$GatewayFQDN,
        [string]$ServerIP,
        [string]$UserGroup
    )
    

    # 自动获取本地 DNS 名称
    if (-not $GatewayFQDN) {
        $hostName = [System.Net.Dns]::GetHostName()
        $domainName = [System.Net.NetworkInformation.IPGlobalProperties]::GetIPGlobalProperties().DomainName
        if ($domainName -and $domainName -ne "") {
            $GatewayFQDN = "$hostName.$domainName"
        } else {
            $GatewayFQDN = $hostName
        }
        Write-Host "检测到本地 DNS 名称: $GatewayFQDN" -ForegroundColor Green
    }
    
    # 自动获取 IP
    if (-not $ServerIP) {
        $ServerIP = (Get-NetIPAddress -AddressFamily IPv4 | Where-Object {
            $_.InterfaceAlias -notlike "*Loopback*" -and $_.IPAddress -ne "127.0.0.1"
        } | Sort-Object InterfaceIndex | Select-Object -First 1).IPAddress
        Write-Host "检测到本机 IP: $ServerIP" -ForegroundColor Green
    }
    
    Write-Host "开始配置 RD 网关..." -ForegroundColor Cyan
    Write-Host "FQDN: $GatewayFQDN" -ForegroundColor White
    Write-Host "IP: $ServerIP" -ForegroundColor White

        # 方法一：使用 if 判断
    if ([string]::IsNullOrWhiteSpace($UserGroup))
    {
        $UserGroup = Read-Host "输入访问网关的用户群组[VDI Users@dora]"
        if ([string]::IsNullOrWhiteSpace($UserGroup)) {
            $UserGroup = "VDI Users@dora"
        }
    }
    Write-Host "用户组: $UserGroup" -ForegroundColor White
    
    # 确认配置
    $confirmation = Read-Host "`n是否继续配置? (Y/N)"
    if ($confirmation -notmatch '^[Yy]') {
        Write-Host "配置已取消。" -ForegroundColor Yellow
        return
    }
    
    try {
        # 1. 安装角色
        Write-Host "`n[1/4] 安装 RD 网关角色..." -ForegroundColor Green
        $feature = Get-WindowsFeature -Name RDS-Gateway
        if ($feature.Installed) {
            Write-Host "  ✓ RD 网关角色已安装" -ForegroundColor Blue
        } else {
            Install-WindowsFeature -Name RDS-Gateway -IncludeManagementTools | Out-Null
            Write-Host "  ✓ RD 网关角色安装成功" -ForegroundColor Green
        }
        
        # 2. 导入模块
        Write-Host "`n[2/4] 导入远程桌面服务模块..." -ForegroundColor Green
        Import-Module RemoteDesktopServices -ErrorAction Stop
        Write-Host "  ✓ 模块导入成功" -ForegroundColor Green
        
        # 3. 配置证书
        Write-Host "`n[3/4] 配置证书..." -ForegroundColor Green
        $cert = New-SelfSignedCertificate -DnsName $GatewayFQDN, $ServerIP -CertStoreLocation "Cert:\LocalMachine\My" -ErrorAction Stop
        Set-Item -Path "RDS:\GatewayServer\SSLCertificate\Thumbprint" -Value $cert.Thumbprint -ErrorAction Stop
        Write-Host "  ✓ 证书配置完成 (指纹: $($cert.Thumbprint.Substring(0, 10))...)" -ForegroundColor Green
        
        # 4. 创建策略和重启服务
        Write-Host "`n[4/4] 创建授权策略并重启服务..." -ForegroundColor Green
        New-Item -Path 'RDS:\GatewayServer\CAP' -Name 'VDI-CAP' -UserGroups $UserGroup -AuthMethod '1' -ErrorAction Stop | Out-Null
        Write-Host "  ✓ CAP 创建成功" -ForegroundColor Green
        
        New-Item -Path 'RDS:\GatewayServer\RAP' -Name 'VDI-RAP' -UserGroups $UserGroup -ComputerGroupType '2' -ErrorAction Stop | Out-Null
        Write-Host "  ✓ RAP 创建成功" -ForegroundColor Green
        
        Restart-Service TSGateway -Force -ErrorAction Stop
        Write-Host "  ✓ 服务重启完成" -ForegroundColor Green
        
        Write-Host "`n🎉 RD 网关配置完成！" -ForegroundColor Green
        Write-Host "访问地址: https://$GatewayFQDN" -ForegroundColor Cyan
    }
    catch {
        Write-Host "`n❌ 配置失败: $($_.Exception.Message)" -ForegroundColor Red
    }
}

do {
    Show-Menu
    $choice = Read-Host "请选择操作"
    switch ($choice) {
        '1' { Set-Network }
        '2' { Join-ADDomain }
        '3' { Install-RDSRoles }        
        '4' { Install-RDPWrapper }
        '5' { Add-VDIUserGroup }
        '6' { Install-Configure-RDGateway  }
    }
    if ($choice -notin 'Q','q') { Read-Host "按 Enter 继续..." }
} while ($choice -notin 'Q','q')